import math
import tkinter as tk
from tkinter import messagebox, ttk
from typing import Optional, Sequence

try:
    from colorstimuli import xyz2colorname, xyz_to_colorstimulus  # type: ignore
except ImportError:  # pragma: no cover
    from .colorstimuli import xyz2colorname, xyz_to_colorstimulus  # type: ignore

try:
    from config import global_config  # type: ignore
except ImportError:  # pragma: no cover
    try:
        from .config import global_config  # type: ignore
    except ImportError:  # pragma: no cover
        global_config = None  # type: ignore

try:
    from util import XYZW_D65  # type: ignore
except ImportError:  # pragma: no cover
    try:
        from .util import XYZW_D65  # type: ignore
    except ImportError:  # pragma: no cover
        XYZW_D65 = [95.047, 100.0, 108.883]


def _get_default_xyzw() -> Sequence[float]:
    if global_config is not None and hasattr(global_config, "xyzw"):
        return tuple(float(v) for v in global_config.xyzw)
    return tuple(float(v) for v in XYZW_D65)


class XYZColorNameApp(tk.Tk):
    def __init__(self) -> None:
        super().__init__()
        self.title("XYZ -> Color Name")
        self.resizable(False, False)

        self.default_xyzw = _get_default_xyzw()
        self.xyz_vars = {
            "X": tk.StringVar(value=""),
            "Y": tk.StringVar(value=""),
            "Z": tk.StringVar(value=""),
        }
        self.xyzw_vars = {
            "Xw": tk.StringVar(),
            "Yw": tk.StringVar(),
            "Zw": tk.StringVar(),
        }
        self.result_vars = {
            "color_name": tk.StringVar(value="-"),
            "ncs_name": tk.StringVar(value="-"),
            "depth_description": tk.StringVar(value="-"),
            "u_name": tk.StringVar(value="-"),
            "lab_value": tk.StringVar(value="-"),
            "jch_value": tk.StringVar(value="-"),
        }

        self._build_ui()
        self._reset_whitepoint()
        self._update_preview_color((0.0, 0.0, 0.0))

    def _build_ui(self) -> None:
        padding = {"padx": 10, "pady": 6}

        input_frame = ttk.LabelFrame(self, text="Input XYZ")
        input_frame.grid(row=0, column=0, sticky="ew", **padding)

        for idx, (label_text, var) in enumerate(self.xyz_vars.items()):
            ttk.Label(input_frame, text=label_text).grid(row=idx, column=0, sticky="w", padx=6, pady=2)
            ttk.Entry(input_frame, textvariable=var, width=12).grid(row=idx, column=1, padx=6, pady=2)

        white_frame = ttk.LabelFrame(self, text="Reference White XYZn (optional)")
        white_frame.grid(row=1, column=0, sticky="ew", **padding)

        for idx, (label_text, var) in enumerate(self.xyzw_vars.items()):
            ttk.Label(white_frame, text=label_text).grid(row=idx, column=0, sticky="w", padx=6, pady=2)
            ttk.Entry(white_frame, textvariable=var, width=12).grid(row=idx, column=1, padx=6, pady=2)

        ttk.Button(white_frame, text="Use Default Whitepoint", command=self._reset_whitepoint).grid(
            row=3, column=0, columnspan=2, pady=(8, 2)
        )

        action_frame = ttk.Frame(self)
        action_frame.grid(row=2, column=0, sticky="ew", **padding)

        ttk.Button(action_frame, text="Compute Color Name", command=self._on_calculate).grid(
            row=0, column=0, padx=6, pady=2
        )

        content_frame = ttk.Frame(self)
        content_frame.grid(row=3, column=0, sticky="ew", **padding)

        self.preview_canvas = tk.Canvas(content_frame, width=120, height=80, bd=1, relief="solid")
        self.preview_canvas.grid(row=0, column=0, rowspan=4, padx=(0, 12))

        result_frame = ttk.LabelFrame(content_frame, text="Results")
        result_frame.grid(row=0, column=1, sticky="nsew")

        result_items = [
            ("Color Name", "color_name"),
            ("NCS Hue", "ncs_name"),
            ("Depth", "depth_description"),
            ("u_name", "u_name"),
            ("Lab*", "lab_value"),
            ("JCh", "jch_value"),
        ]

        for idx, (label_text, key) in enumerate(result_items):
            ttk.Label(result_frame, text=label_text + ":").grid(row=idx, column=0, sticky="w", padx=6, pady=2)
            ttk.Label(result_frame, textvariable=self.result_vars[key], width=32).grid(
                row=idx, column=1, sticky="w", padx=6, pady=2
            )

    def _reset_whitepoint(self) -> None:
        for value, var in zip(self.default_xyzw, self.xyzw_vars.values()):
            var.set(f"{value:.3f}")

    def _parse_xyz_values(self) -> Optional[Sequence[float]]:
        try:
            values = [float(var.get()) for var in self.xyz_vars.values()]
        except ValueError:
            messagebox.showerror("Invalid input", "Please enter numeric values for X, Y, Z.")
            return None
        return values

    def _parse_whitepoint(self) -> Optional[Sequence[float]]:
        entries = [var.get().strip() for var in self.xyzw_vars.values()]
        if not any(entries):
            return None
        try:
            values = [float(value) for value in entries]
        except ValueError:
            messagebox.showerror("Invalid input", "Reference white must be numeric or left blank.")
            return None
        if len(values) != 3:
            messagebox.showerror("Invalid input", "Please provide all of Xw, Yw, Zw for the whitepoint.")
            return None
        return values

    def _on_calculate(self) -> None:
        xyz = self._parse_xyz_values()
        if xyz is None:
            return
        xyzw = self._parse_whitepoint()

        try:
            config_kwargs = {}
            if global_config is not None:
                config_kwargs["config"] = global_config
            info = xyz2colorname(
                xyz,
                xyzw=xyzw,
                include_details=True,
                **config_kwargs,
            )
            stimulus = xyz_to_colorstimulus(
                xyz,
                xyzw=xyzw,
                **config_kwargs,
            )
        except Exception as exc:  # pylint: disable=broad-except
            messagebox.showerror("Computation failed", str(exc))
            return

        appearance = stimulus.appearance
        srgb_rep = stimulus.display_representations.get("sRGB")

        self.result_vars["color_name"].set(info.get("color_name", "-"))
        self.result_vars["ncs_name"].set(info.get("ncs_name", "-"))
        self.result_vars["depth_description"].set(info.get("depth_description", "-"))
        self.result_vars["u_name"].set(info.get("u_name", "-"))

        if appearance is not None:
            lab = ", ".join(f"{value:.2f}" for value in appearance.lab_value)
            jch = ", ".join(f"{value:.2f}" for value in appearance.JCh)
            self.result_vars["lab_value"].set(lab)
            self.result_vars["jch_value"].set(jch)
        else:
            self.result_vars["lab_value"].set("-")
            self.result_vars["jch_value"].set("-")

        if srgb_rep is not None:
            self._update_preview_color(srgb_rep.rgb_values)
        else:
            self._update_preview_color((0.0, 0.0, 0.0))

    def _update_preview_color(self, rgb: Sequence[float]) -> None:
        r, g, b = [max(0, min(1, float(value))) for value in rgb]
        hex_color = "#%02x%02x%02x" % (
            math.floor(r * 255),
            math.floor(g * 255),
            math.floor(b * 255),
        )
        self.preview_canvas.delete("all")
        self.preview_canvas.create_rectangle(0, 0, 120, 80, fill=hex_color, outline=hex_color)


def main() -> None:
    app = XYZColorNameApp()
    app.mainloop()


if __name__ == "__main__":
    main()
