import sys
from pathlib import Path
sys.path.append(str(Path(__file__).resolve().parents[1]))
import unittest

import numpy as np

from colorstimuli import xyz_to_colorstimulus, xyz2colorname
from display import DisplayModel


class ColorStimuliTests(unittest.TestCase):
    @classmethod
    def setUpClass(cls) -> None:
        cls.display = DisplayModel("srgb")

    def test_xyz_to_colorstimulus_primary_red(self) -> None:
        xyz = self.display.rgb_to_xyz(np.array([1.0, 0.0, 0.0]))[0]
        stimulus = xyz_to_colorstimulus(xyz)
        appearance = stimulus.appearance
        self.assertIsNotNone(appearance)
        assert appearance is not None  # type checker guard
        self.assertEqual(appearance.classification, "Red")
        self.assertEqual(appearance.NCS_name, "Y80R")
        self.assertEqual(appearance.depth_description, "Very Deep")

    def test_xyz_to_colorstimulus_primary_green(self) -> None:
        xyz = self.display.rgb_to_xyz(np.array([0.0, 1.0, 0.0]))[0]
        stimulus = xyz_to_colorstimulus(xyz)
        appearance = stimulus.appearance
        self.assertIsNotNone(appearance)
        assert appearance is not None
        self.assertEqual(appearance.classification, "Green")
        self.assertEqual(appearance.NCS_name, "G17Y")

    def test_xyz2colorname_basic_output(self) -> None:
        xyz = self.display.rgb_to_xyz(np.array([0.0, 0.0, 1.0]))[0]
        name_only = xyz2colorname(xyz)
        self.assertEqual(name_only, "Purple")

        details = xyz2colorname(xyz, include_details=True)
        self.assertIsInstance(details, dict)
        self.assertEqual(details["color_name"], "Purple")
        self.assertIn("ncs_name", details)
        self.assertIn("lab_value", details)
        self.assertIn("JCh", details)
        self.assertIn("rgb_preview", details)

    def test_xyz_to_colorstimulus_invalid_length(self) -> None:
        with self.assertRaises(ValueError):
            xyz_to_colorstimulus([1.0, 2.0])


if __name__ == "__main__":
    unittest.main()
