import uuid
import time
import math
from typing import List, Dict, Any, Optional
from pydantic import BaseModel, Field

# --- Nested Data Structures ---

class SpectralData(BaseModel):
    """Stores the raw spectral curve data from a measurement."""
    points_count: int = Field(..., description="The number of data points in the curve.")
    interval_nm: int = Field(..., description="The wavelength step between points (e.g., 10 nm).")
    start_wavelength_nm: int = Field(..., description="The starting wavelength for the measurement (e.g., 400 nm).")
    reflectance_values: List[float] = Field(..., description="The list of reflectance values.")

class DisplayRepresentation(BaseModel):
    """
    Stores the color's appearance in a specific RGB color space.
    This is used for accurate on-screen display.
    """
    color_space_name: str = Field(..., description="The name of the target RGB space (e.g., 'sRGB', 'Display P3').")
    rgb_values: List[float] = Field(..., description="The [R, G, B] values, scaled from 0.0 to 1.0.")
    is_out_of_gamut: bool = Field(..., description="True if the color is outside the target color space's gamut.")

# --- Main Data Structure (V4.0) ---

class AppearanceData(BaseModel):
    """
    Stores derived appearance attributes, calculated from the scientific core.
    This is the source for generating the 'u_name'.
    """
    lab_value: List[float] = Field(..., description="The calculated [L*, a*, b*] values.")
    JCh: List[float] = Field(..., description="The calculated [Lightness J, Chroma C, Hue Angle h] values.")
    NCS_name: str = Field(..., description="The NCS Hue Name (e.g., 'Y90R').")
    depth_description: str = Field(..., description="The color depth description (e.g., 'deep').")
    classification: str = Field(..., description="The basic color category (e.g., 'Red').")

class ScientificData(BaseModel):
    """
    Stores the absolute scientific definition of the color, including observation conditions.
    This is the 'single source of truth'.
    """
    xyz_value: List[float] = Field(..., description="The core [X, Y, Z] tristimulus values.")
    observer_angle: int = Field(..., description="The observer viewing angle (2 or 10 degrees).")
    reference_white_xyz: List[float] = Field(..., description="The [X, Y, Z] values of the reference white point.")
    
    # Appearance model parameters
    adapting_luminance_La: float = Field(..., description="The adapting luminance in cd/m^2.")
    background_luminance_Yb: float = Field(..., description="The luminance factor of the background.")
    surround_condition: str = Field(..., description="The viewing surround (e.g., 'avg', 'dim', 'dark').")

class SourceInfo(BaseModel):
    """Stores information about where the color data came from."""
    type: str = Field(..., description="The origin type, e.g., 'QTX_FILE', 'COLOR_PICKER'.")
    origin_identifier: Optional[str] = Field(None, description="The origin name, e.g., a filename.")
    s_name: Optional[str] = Field(None, description="The source name, from the 'STD_NAME' in a QTX file.")
    spectral_curve: Optional[SpectralData] = Field(None, description="The raw spectral data, if the source was a measurement.")

class ColorStimulus(BaseModel):
    """
    The main, unified color object for the entire application.
    This is the definitive V4.0 structure.
    """
    id: str = Field(default_factory=lambda: str(uuid.uuid4()), description="A machine-readable unique ID (UUID).")
    u_name: Optional[str] = Field(None, description="A human-readable, system-wide unique name based on appearance.")
    
    source: SourceInfo
    scientific_core: ScientificData
    metadata: Dict[str, Any] = Field(default_factory=dict, description="A flexible dictionary for all other descriptive metadata (e.g., aperture, substrate).")
    
    # Derived data, calculated on demand
    appearance: Optional[AppearanceData] = None
    display_representations: Dict[str, DisplayRepresentation] = Field(default_factory=dict)


# --- Helper Function for u_name Generation ---

def generate_u_name(appearance: AppearanceData) -> str:
    """Generates the universal name (u_name) from appearance data."""
    
    # Format J, C, h as zero-padded 3-digit strings
    J_int = math.floor(appearance.JCh[0])
    C_int = math.floor(appearance.JCh[1])
    h_int = math.floor(appearance.JCh[2])
    
    j_str = f"{J_int:03d}"
    c_str = f"{C_int:03d}"
    h_str = f"{h_int:03d}"
    
    # Get H, D, Name and remove any spaces
    h_name_str = appearance.NCS_name.replace(" ", "")
    d_str = appearance.depth_description.replace(" ", "")
    name_str = appearance.classification.replace(" ", "")

    # Combine all parts
    u_name = f"{j_str}{c_str}{h_str}{h_name_str}{d_str}{name_str}"
    
    return u_name