import uuid
from typing import Any, Dict, Optional, Sequence
import numpy as np
import colour

try:
    from .models import ColorStimulus, ScientificData, AppearanceData, DisplayRepresentation, SourceInfo, generate_u_name
    from .util import scam_to_xyz, xyz_to_scam
    from .colorname import get_ncs_hue_name, get_depth_description, classify_cielab_colors
    from .display import DisplayModel
    from .config import global_config
except ImportError:  # pragma: no cover
    from models import ColorStimulus, ScientificData, AppearanceData, DisplayRepresentation, SourceInfo, generate_u_name
    from util import scam_to_xyz, xyz_to_scam
    from colorname import get_ncs_hue_name, get_depth_description, classify_cielab_colors
    from display import DisplayModel
    try:
        from config import global_config
    except ImportError:
        class _DefaultConfig:
            xyzw = [95.047, 100.0, 108.883]
            la = 64.0
            yb = 20.0
            surround = 'avg'

        global_config = _DefaultConfig()

def jch_to_colorstimulus(
    jch: list,
    xyzw: Optional[list] = None,
    la: Optional[float] = None,
    yb: Optional[float] = None,
    surround: Optional[str] = None,
    viewing_condition: Optional[Any] = None,
    config=global_config
) -> ColorStimulus:
    """
    严格参考qtx.py:_create_stimulus_from_data实现，将sCAM JCh及相关参数转换为ColorStimulus对象。
    支持通过 config 统一管理参数，外部传参可覆盖。
    """
    # 参数优先级：显式传参 > config
    xyzw = xyzw if xyzw is not None else config.xyzw
    la = la if la is not None else config.la
    yb = yb if yb is not None else config.yb
    surround = surround if surround is not None else config.surround

    # 1. JCh → XYZ
    xyz = scam_to_xyz(jch, xyzw, yb, la, surround)
    xyz = np.asarray(xyz).flatten()

    # 2. XYZ → Lab，参考白点需先转xyY
    xyY = colour.XYZ_to_xyY(np.asarray(xyzw))
    lab = colour.XYZ_to_Lab(xyz, illuminant=xyY)
    lab = np.asarray(lab).tolist()

    # 3. appearance: JCh, H, D, NCS_name, depth_description, classification
    jch_full, hd = xyz_to_scam(
        xyz=xyz,
        xyz_w=np.asarray(xyzw),
        y_b=yb,
        l_a=la,
        surround=surround,
        mode='full'
    )
    H_val, D_val = hd[0], hd[1]
    appearance = AppearanceData(
        lab_value=lab,
        JCh=jch_full.tolist(),
        NCS_name=get_ncs_hue_name(H_val),
        depth_description=get_depth_description(D_val),
        classification=classify_cielab_colors(lab)
    )

    # 4. scientific_core
    scientific_core = ScientificData(
        xyz_value=xyz.tolist(),
        observer_angle=10,
        reference_white_xyz=xyzw,
        adapting_luminance_La=la,
        background_luminance_Yb=yb,
        surround_condition=surround
    )

    # 5. display_representations
    srgb_model = DisplayModel('srgb')
    rgb_values = srgb_model.xyz_to_rgb(np.array([xyz]))[0]

    # 新的超色域判断逻辑
    # 1. xyz 有分量 <0 且有分量 > 参考白点
    xyz_w = np.asarray(xyzw).flatten()
    xyz_below_zero = np.any(xyz < 0)
    xyz_above_white = np.any(xyz > xyz_w)
    xyz_out_of_gamut = xyz_below_zero or xyz_above_white

    # 2. rgb 有分量 <0 或 >1
    rgb_out_of_gamut = np.any(rgb_values <= 0) or np.any(rgb_values > 1)

    is_out_of_gamut = bool(xyz_out_of_gamut or rgb_out_of_gamut)
    # rgb_values 最终clip到0-1
    rgb_values_clipped = np.clip(rgb_values, 0, 1)

    display_rep = DisplayRepresentation(
        color_space_name='sRGB',
        rgb_values=rgb_values_clipped.tolist(),
        is_out_of_gamut=is_out_of_gamut
    )
    display_representations = {'sRGB': display_rep}

    # 6. 其它字段
    color_id = str(uuid.uuid4())
    u_name = generate_u_name(appearance)
    metadata: Dict[str, Any] = {}
    source = SourceInfo(type="JCH_INPUT", origin_identifier=None)

    # 7. 组装 ColorStimulus
    color_stimulus = ColorStimulus(
        id=color_id,
        u_name=u_name,
        source=source,
        scientific_core=scientific_core,
        metadata=metadata,
        appearance=appearance,
        display_representations=display_representations
    )
    return color_stimulus

def lab_to_colorstimulus(
    lab: list,
    xyzw: Optional[list] = None,
    la: Optional[float] = None,
    yb: Optional[float] = None,
    surround: Optional[str] = None,
    viewing_condition: Optional[Any] = None,
    config=global_config
) -> ColorStimulus:
    """
    将 CIELAB L*a*b* 值转换为 ColorStimulus 对象。
    步骤： Lab → XYZ → sCAM JCh 等派生数据，逻辑与 jch_to_colorstimulus 保持一致。
    """
    # 参数优先级：显式传参 > config
    xyzw = xyzw if xyzw is not None else config.xyzw
    la = la if la is not None else config.la
    yb = yb if yb is not None else config.yb
    surround = surround if surround is not None else config.surround

    # 1. Lab → XYZ
    xyY_w = colour.XYZ_to_xyY(np.asarray(xyzw))
    xyz = colour.Lab_to_XYZ(lab, illuminant=xyY_w)
    xyz = np.asarray(xyz).flatten()

    # 2. XYZ → JCh (sCAM full)
    jch_full, hd = xyz_to_scam(
        xyz=xyz,
        xyz_w=np.asarray(xyzw),
        y_b=yb,
        l_a=la,
        surround=surround,
        mode='full'
    )
    H_val, D_val = hd[0], hd[1]

    appearance = AppearanceData(
        lab_value=list(lab),
        JCh=jch_full.tolist(),
        NCS_name=get_ncs_hue_name(H_val),
        depth_description=get_depth_description(D_val),
        classification=classify_cielab_colors(lab)
    )

    # 3. scientific_core
    scientific_core = ScientificData(
        xyz_value=xyz.tolist(),
        observer_angle=10,
        reference_white_xyz=xyzw,
        adapting_luminance_La=la,
        background_luminance_Yb=yb,
        surround_condition=surround
    )

    # 4. display_representations (sRGB)
    srgb_model = DisplayModel('srgb')
    rgb_values = srgb_model.xyz_to_rgb(np.array([xyz]))[0]

    xyz_w = np.asarray(xyzw).flatten()
    xyz_out_of_gamut = np.any(xyz < 0) and np.any(xyz > xyz_w)
    rgb_out_of_gamut = np.any(rgb_values < 0) or np.any(rgb_values > 1)
    is_out_of_gamut = bool(xyz_out_of_gamut or rgb_out_of_gamut)
    rgb_values_clipped = np.clip(rgb_values, 0, 1)

    display_rep = DisplayRepresentation(
        color_space_name='sRGB',
        rgb_values=rgb_values_clipped.tolist(),
        is_out_of_gamut=is_out_of_gamut
    )
    display_representations = {'sRGB': display_rep}

    # 5. 其它字段
    color_id = str(uuid.uuid4())
    u_name = generate_u_name(appearance)
    metadata: Dict[str, Any] = {}
    source = SourceInfo(type="LAB_INPUT", origin_identifier=None)

    # 6. 组装 ColorStimulus
    return ColorStimulus(
        id=color_id,
        u_name=u_name,
        source=source,
        scientific_core=scientific_core,
        metadata=metadata,
        appearance=appearance,
        display_representations=display_representations
    )

def srgb_to_colorstimulus(
    rgb: list,
    xyzw: Optional[list] = None,
    la: Optional[float] = None,
    yb: Optional[float] = None,
    surround: Optional[str] = None,
    viewing_condition: Optional[Any] = None,
    config=global_config
) -> ColorStimulus:
    """
    将 sRGB 值转换为 ColorStimulus 对象。
    步骤： sRGB → XYZ → sCAM JCh 等派生数据，逻辑与 jch_to_colorstimulus / lab_to_colorstimulus 保持一致。
    rgb 输入应为 0-1 范围的 [R, G, B]。
    """
    # 参数优先级：显式传参 > config
    xyzw = xyzw if xyzw is not None else config.xyzw
    la = la if la is not None else config.la
    yb = yb if yb is not None else config.yb
    surround = surround if surround is not None else config.surround

    # 1. sRGB → XYZ
    srgb_model = DisplayModel('srgb')
    rgb_arr = np.asarray(rgb).reshape(1, 3)
    xyz = srgb_model.rgb_to_xyz(rgb_arr)[0]

    # 2. XYZ → Lab
    xyY_w = colour.XYZ_to_xyY(np.asarray(xyzw))
    lab = colour.XYZ_to_Lab(xyz, illuminant=xyY_w)
    lab = np.asarray(lab).tolist()

    # 3. XYZ → JCh (sCAM full)
    jch_full, hd = xyz_to_scam(
        xyz=xyz,
        xyz_w=np.asarray(xyzw),
        y_b=yb,
        l_a=la,
        surround=surround,
        mode='full'
    )
    H_val, D_val = hd[0], hd[1]

    appearance = AppearanceData(
        lab_value=lab,
        JCh=jch_full.tolist(),
        NCS_name=get_ncs_hue_name(H_val),
        depth_description=get_depth_description(D_val),
        classification=classify_cielab_colors(lab)
    )

    # 4. scientific_core
    scientific_core = ScientificData(
        xyz_value=xyz.tolist(),
        observer_angle=10,
        reference_white_xyz=xyzw,
        adapting_luminance_La=la,
        background_luminance_Yb=yb,
        surround_condition=surround
    )

    # 5. display_representations (sRGB original)
    rgb_values = np.clip(rgb, 0, 1)
    xyz_w = np.asarray(xyzw).flatten()
    xyz_out_of_gamut = np.any(xyz < 0) and np.any(xyz > xyz_w)
    is_out_of_gamut = bool(xyz_out_of_gamut)

    display_rep = DisplayRepresentation(
        color_space_name='sRGB',
        rgb_values=rgb_values if isinstance(rgb_values, list) else rgb_values.tolist(),
        is_out_of_gamut=is_out_of_gamut
    )
    display_representations = {'sRGB': display_rep}

    # 6. 其它字段
    color_id = str(uuid.uuid4())
    u_name = generate_u_name(appearance)
    metadata: Dict[str, Any] = {}
    source = SourceInfo(type="SRGB_INPUT", origin_identifier=None)

    # 7. 组装 ColorStimulus
    return ColorStimulus(
        id=color_id,
        u_name=u_name,
        source=source,
        scientific_core=scientific_core,
    metadata=metadata,
    appearance=appearance,
    display_representations=display_representations
)
def xyz_to_colorstimulus(
    xyz: Sequence[float] | np.ndarray,
    xyzw: Optional[Sequence[float]] = None,
    la: Optional[float] = None,
    yb: Optional[float] = None,
    surround: Optional[str] = None,
    viewing_condition: Optional[Any] = None,
    config=global_config
) -> ColorStimulus:
    """
    将单个 XYZ 三刺激值直接转换为 ColorStimulus 对象。
    支持外部覆盖白点 (xyzw)、La、Yb、surround 等观看条件。
    """
    xyz_arr = np.asarray(xyz, dtype=np.float64).flatten()
    if xyz_arr.size != 3:
        raise ValueError("xyz must contain exactly three components.")
    if np.any(np.isnan(xyz_arr)):
        raise ValueError("xyz contains NaN values.")

    xyzw_arr = np.asarray(xyzw if xyzw is not None else config.xyzw, dtype=np.float64).flatten()
    if xyzw_arr.size != 3:
        raise ValueError("xyzw must contain exactly three components.")

    la_val = float(la if la is not None else config.la)
    yb_val = float(yb if yb is not None else config.yb)
    surround_val = surround if surround is not None else config.surround

    xyY_w = colour.XYZ_to_xyY(xyzw_arr)
    lab = colour.XYZ_to_Lab(xyz_arr, illuminant=xyY_w)
    lab_list = np.asarray(lab).tolist()

    jch_full, hd = xyz_to_scam(
        xyz=xyz_arr,
        xyz_w=xyzw_arr,
        y_b=yb_val,
        l_a=la_val,
        surround=surround_val,
        mode='full'
    )
    H_val, D_val = hd[0], hd[1]

    appearance = AppearanceData(
        lab_value=lab_list,
        JCh=jch_full.tolist(),
        NCS_name=get_ncs_hue_name(H_val),
        depth_description=get_depth_description(D_val),
        classification=classify_cielab_colors(lab_list)
    )

    scientific_core = ScientificData(
        xyz_value=xyz_arr.tolist(),
        observer_angle=10,
        reference_white_xyz=xyzw_arr.tolist(),
        adapting_luminance_La=la_val,
        background_luminance_Yb=yb_val,
        surround_condition=surround_val
    )

    srgb_model = DisplayModel('srgb')
    rgb_values = srgb_model.xyz_to_rgb(np.array([xyz_arr]))[0]

    xyz_below_zero = np.any(xyz_arr < 0)
    xyz_above_white = np.any(xyz_arr > xyzw_arr)
    xyz_out_of_gamut = xyz_below_zero or xyz_above_white

    rgb_out_of_gamut = np.any(rgb_values <= 0) or np.any(rgb_values > 1)
    is_out_of_gamut = bool(xyz_out_of_gamut or rgb_out_of_gamut)
    rgb_values_clipped = np.clip(rgb_values, 0, 1)

    display_rep = DisplayRepresentation(
        color_space_name='sRGB',
        rgb_values=rgb_values_clipped.tolist(),
        is_out_of_gamut=is_out_of_gamut
    )
    display_representations = {'sRGB': display_rep}

    color_id = str(uuid.uuid4())
    u_name = generate_u_name(appearance)
    metadata: Dict[str, Any] = {}
    source = SourceInfo(type="XYZ_INPUT", origin_identifier=None)

    return ColorStimulus(
        id=color_id,
        u_name=u_name,
        source=source,
        scientific_core=scientific_core,
        metadata=metadata,
        appearance=appearance,
        display_representations=display_representations
    )


def xyz2colorname(
    xyz: Sequence[float] | np.ndarray,
    xyzw: Optional[Sequence[float]] = None,
    la: Optional[float] = None,
    yb: Optional[float] = None,
    surround: Optional[str] = None,
    config=global_config,
    include_details: bool = False
) -> Dict[str, Any] | str:
    """
    直接从 XYZ 获取配套的颜色名称信息。
    默认返回基础色名；include_details=True 时返回包含 NCS、深度描述等全部信息的字典。
    """
    stimulus = xyz_to_colorstimulus(
        xyz=xyz,
        xyzw=xyzw,
        la=la,
        yb=yb,
        surround=surround,
        config=config
    )
    appearance = stimulus.appearance
    if appearance is None:
        raise ValueError("Appearance data is missing from ColorStimulus.")

    if not include_details:
        return appearance.classification

    srgb_rep = stimulus.display_representations.get('sRGB')
    return {
        'color_name': appearance.classification,
        'ncs_name': appearance.NCS_name,
        'depth_description': appearance.depth_description,
        'lab_value': appearance.lab_value,
        'JCh': appearance.JCh,
        'u_name': stimulus.u_name,
        'is_out_of_gamut': srgb_rep.is_out_of_gamut if srgb_rep else None,
        'rgb_preview': srgb_rep.rgb_values if srgb_rep else None,
    }


def create_colorstimulus_from_xyz(
    xyz_values: Sequence[float] | np.ndarray,
    xyzw: Optional[Sequence[float]] = None,
    la: Optional[float] = None,
    yb: Optional[float] = None,
    surround: Optional[str] = None,
    viewing_condition: Optional[Any] = None,
    config=global_config
) -> ColorStimulus:
    """
    向后兼容的封装，等价于调用 xyz_to_colorstimulus。
    """
    return xyz_to_colorstimulus(
        xyz=xyz_values,
        xyzw=xyzw,
        la=la,
        yb=yb,
        surround=surround,
        viewing_condition=viewing_condition,
        config=config
    )
def test_jch_to_colorstimulus():
    # 测试用JCh（C<40），可选取不同色相
    test_jch_list = [
        [0, 0, 0],   # 低C，偏红
        [60, 25, 90],   # 低C，偏黄
        [40, 35, 200],  # 低C，偏绿
        [70, 15, 300],  # 低C，偏蓝
    ]
    # 参考白点（D65），La、Yb、surround与qtx.py一致
    for idx, jch in enumerate(test_jch_list):
        stim = jch_to_colorstimulus(jch)
        print(f"\n--- Test Case #{idx+1} ---")
        print(f"u_name: {stim.u_name}")
        print(f"Lab: {stim.appearance.lab_value if stim.appearance else 'N/A'}")
        print(f"JCh: {stim.appearance.JCh if stim.appearance else 'N/A'}")
        print(f"NCS_name: {stim.appearance.NCS_name if stim.appearance else 'N/A'}")
        print(f"depth_description: {stim.appearance.depth_description if stim.appearance else 'N/A'}")
        print(f"classification: {stim.appearance.classification if stim.appearance else 'N/A'}")
        print(f"display_representations: {stim.display_representations if stim.display_representations else 'N/A'}")

def test_lab_to_colorstimulus():
    test_lab_list = [
        [50, 0, 0],     # 中性灰
        [70, 20, 40],   # 偏红黄
        [30, -20, 20],  # 偏绿蓝
        [80, 0, -40],   # 偏蓝
    ]
    for idx, lab in enumerate(test_lab_list):
        stim = lab_to_colorstimulus(lab)
        print(f"\n--- Lab Test Case #{idx+1} ---")
        print(f"u_name: {stim.u_name}")
        print(f"Lab: {stim.appearance.lab_value if stim.appearance else 'N/A'}")
        print(f"JCh: {stim.appearance.JCh if stim.appearance else 'N/A'}")
        print(f"NCS_name: {stim.appearance.NCS_name if stim.appearance else 'N/A'}")
        print(f"depth_description: {stim.appearance.depth_description if stim.appearance else 'N/A'}")
        print(f"classification: {stim.appearance.classification if stim.appearance else 'N/A'}")
        print(f"display_representations: {stim.display_representations if stim.display_representations else 'N/A'}")

def test_srgb_to_colorstimulus():
    test_rgb_list = [
        [0.0, 0.0, 0.0],   # Black
        [1.0, 1.0, 1.0],   # White
        [1.0, 0.0, 0.0],   # Red
        [0.0, 1.0, 0.0],   # Green
        [0.0, 0.0, 1.0],   # Blue
        [0.5, 0.5, 0.5],   # Grey
    ]
    for idx, rgb in enumerate(test_rgb_list):
        stim = srgb_to_colorstimulus(rgb)
        print(f"\n--- sRGB Test Case #{idx+1} ---")
        print(f"u_name: {stim.u_name}")
        print(f"Lab: {stim.appearance.lab_value if stim.appearance else 'N/A'}")
        print(f"JCh: {stim.appearance.JCh if stim.appearance else 'N/A'}")
        print(f"NCS_name: {stim.appearance.NCS_name if stim.appearance else 'N/A'}")
        print(f"depth_description: {stim.appearance.depth_description if stim.appearance else 'N/A'}")
        print(f"classification: {stim.appearance.classification if stim.appearance else 'N/A'}")
        print(f"display_representations: {stim.display_representations if stim.display_representations else 'N/A'}")

def test_xyz2colorname():
    test_xyz_list = [
        [41.24, 21.26, 1.93],   # 较暗红
        [18.05, 45.00, 50.00],  # 偏青绿
        [95.05, 100.0, 108.88], # D65 白点
    ]
    for idx, xyz in enumerate(test_xyz_list):
        info = xyz2colorname(xyz, include_details=True)
        print(f"\n--- XYZ Test Case #{idx+1} ---")
        for key, value in info.items():
            print(f"{key}: {value}")

if __name__ == "__main__":
    test_jch_to_colorstimulus()
    test_lab_to_colorstimulus()
    test_srgb_to_colorstimulus()
    test_xyz2colorname()
