import math

def get_depth_description(D: float) -> str:
    """
    根据输入的深度值 D 返回相应的描述性文本。

    Args:
        D: 一个代表深度的数值。

    Returns:
        一个描述深度的字符串。
    """
    if D < 20:
        desc = 'Very pale'  # 非常浅
    elif D < 40:
        desc = 'Pale'  # 浅
    elif D < 60:
        desc = 'Mid depth'  # 有点深
    elif D < 80:
        desc = 'Deep'  # 深
    else:
        desc = 'Very Deep'  # 非常深
    return desc

def get_ncs_hue_name(H: float) -> str:
    """
    Converts a numerical hue value (from a 400-degree circle) into an NCS hue name.

    Args:
        H: A numerical hue value. The function will normalize it to the [0, 400) range.

    Returns:
        The corresponding NCS (Natural Color System) hue notation as a string.
    """
    # Normalize H to the range [0, 400). Python's % operator handles negative inputs correctly.
    H = H % 400

    # Handle the four primary NCS hues
    if H == 0:
        return 'R'  # Pure Red
    if H == 100:
        return 'Y'  # Pure Yellow
    if H == 200:
        return 'G'  # Pure Green
    if H == 300:
        return 'B'  # Pure Blue

    # Handle the in-between hues based on the quadrant
    if 0 < H < 100:  # Yellow-Red quadrant
        percent = 100 - H
        return f'Y{round(percent)}R'
    elif 100 < H < 200:  # Green-Yellow quadrant
        percent = 200 - H
        return f'G{round(percent)}Y'
    elif 200 < H < 300:  # Blue-Green quadrant
        percent = 300 - H
        return f'B{round(percent)}G'
    else:  # 300 < H < 400, Red-Blue quadrant
        percent = 400 - H
        return f'R{round(percent)}B'

def classify_cielab_colors(lab_data: list[float] | tuple[float, ...]) -> str:
    """
    Classifies a CIELAB color into a common color name.

    Args:
        lab_data: A list or tuple containing the L*, a*, and b* values.
                  Example: [54.29, 80.81, 69.88]

    Returns:
        A string with the classified color name (e.g., 'Red', 'Blue', 'Grey').
    """
    L_star, a_star, b_star = lab_data

    # Calculate Chroma (C*)
    C_star = math.sqrt(a_star**2 + b_star**2)

    # Calculate hue angle in degrees (h) and normalize to the range [0, 360)
    h_deg = math.degrees(math.atan2(b_star, a_star)) % 360

    # 1. Achromatic colors (very low saturation)
    if C_star <= 5:
        if L_star > 89:
            return 'White'
        elif L_star > 50:
            return 'Light Grey'
        elif L_star > 36:
            return 'Dark Grey'
        else:
            return 'Black'

    # 2. Specific chromatic color rules (e.g., Brown, Pink)
    """
    if 29 < L_star <= 60 and C_star <= 44 and 25 <= h_deg <= 80:
        return 'Brown'
    if L_star > 60 and C_star <= 50 and (h_deg >= 340 or h_deg <= 65):
        return 'Pink'

    # 3. General classification based on hue angle
    
    if 15 <= h_deg <= 41:
        return 'Red'
    elif 41 < h_deg <= 70:
        return 'Orange'
    elif 70 < h_deg <= 99:
        return 'Yellow'
    elif 99 < h_deg <= 200:
        return 'Green'
    elif 200 < h_deg <= 244:
        return 'Turquoise'
    elif 244 < h_deg <= 288:
        return 'Blue'
    else:
        return 'Purple'
        """
    #the hue angle boundaries to: red (0-35), orange (35-), yellow (70-),
    #  Green (90-), Tur (190), Blue (244)-Purple (271-360).  Pink (343-40), Brown (20-95).
    """
    if 29 < L_star <= 60 and C_star <= 44 and 20 <= h_deg <= 95:
        return 'Brown'
    if L_star > 60 and C_star <= 50 and (h_deg >= 343 or h_deg <= 40):
        return 'Pink'
    
    if 0 <= h_deg <= 35:
        return 'Red'
    elif 35 < h_deg <= 70:
        return 'Orange'
    elif 70 < h_deg <= 99:
        return 'Yellow'
    elif 99 < h_deg <= 190:
        return 'Green'
    elif 190 < h_deg <= 244:
        return 'Turquoise'
    elif 244 < h_deg <= 271:
        return 'Blue'
    else:
        return 'Purple'
    """
    ###9.8new name
    if 29 < L_star <= 60 and C_star <= 44 and 20 <= h_deg <= 80:
        return 'Brown'
    if L_star > 60 and C_star <= 50 and (h_deg >= 343 or h_deg <= 40):
        return 'Pink'
    
    if 0 <= h_deg <= 41:
        return 'Red'
    elif 41 < h_deg <= 70:
        return 'Orange'
    elif 70 < h_deg <= 99:
        return 'Yellow'
    elif 99 < h_deg <= 200:
        return 'Green'
    elif 200 < h_deg <= 244:
        return 'Turquoise'
    elif 244 < h_deg <= 279:
        return 'Blue'
    else:
        return 'Purple'